// GBox2Eth - Die Maerklin Gleisbox am PC
//
// Die Gleisbox kann mit diesem Programm am PC wie eine CentralStation 2 betrieben werden. Dazu
// werden die CAN-Botschaften der Gleisbox auf Ethernet-UDP-Frames umgesetzt und umgekehrt.
// CAN-Interface: Tiny CAN von MHS-Elektronik
//
// Aufruf:     gbox2eth
//             -> moegliche Aufrufoptionen werden angezeigt
//
// Beispiel:   gbox2eth 5 192.168.0.2
//
// Verwendung: Aufruf auf der Kommandozeile, Programm kann ber Tasteneingaben gesteuert werden
//             Das Tiny-CAN-Interface muss vorher angesteckt worden sein und die Gleisbox laufen
//             Abschluss eines Kommandos mit <Return>, Abbruch mit <Escape>
//
// 2010-06-11  erste Version, getestet mit WinDigipet und RocRail
// 2010-09-05  einfache Kommandozeile, mfx-Unterstuetzung
// 2010-10-10  zweite IP-Adresse ermoeglicht, Umstellung auf Tiny-CAN-Treiber 2.10
// 2011-02-12  Fehlerkorrektur bzgl. UID, Option "passiv" eingebaut
//
// (c) Stefan Krauss, 70437 Stuttgart
// 
// Lizenz: Nutzung und Weitergabe nach der GNU General Public License (GPL), Version 3
//         Es wird keinerlei Haftung oder Gewaehrleistung uebernommen, siehe GPL!

#include "stdafx.h"
#include "cfgfile.h"
#include "cmdline.h"
#include "../../can_api/lib/can_drv.h"

// Globale Variablen
SOCKET sock_out;
SOCKADDR_IN serv_out;
HANDLE udp_mutex;
HANDLE can_mutex;
unsigned int passive_mode; // 1 = passive mode

// Callback-Routine fuer CAN-Empfang
// Die Funktion wird in einem eigenen Thread aufgerufen, wenn eine CAN-Botschaft empfangen wird.
// Sie liest die Botschaft und versendet sie ueber die Ethernet-Schnittstelle als UDP-Frame.

void CALLBACK_TYPE RxEvent(unsigned int index, struct TCanMsg *msg, int count)
{
	struct TCanMsg can_msg;
	char udp_frame[20];
	struct TCanMsg out_msg;
	unsigned long uid;
	unsigned int i;

	while (CanReceive(0, &can_msg, 1) > 0)
	{
		if (can_msg.MsgEFF == 1) // nur extended CAN-Botschaften auswerten
		{
			udp_frame[0] = (can_msg.Id >> 24) & 0x000000FF;
			udp_frame[1] = (can_msg.Id >> 16) & 0x000000FF;
			udp_frame[2] = (can_msg.Id >> 8) & 0x000000FF;
			udp_frame[3] = can_msg.Id & 0x000000FF;
			udp_frame[4] = can_msg.MsgLen;
			for (i = 0; i < 8; ++i)
			{
				if (i < can_msg.MsgLen)
					udp_frame[5+i] = can_msg.MsgData[i];
				else
					udp_frame[5+i] = 0;
			}
			WaitForSingleObject(udp_mutex, INFINITE);
			sendto(sock_out, udp_frame, 13, 0, (struct sockaddr*)&serv_out, sizeof(serv_out));
			ReleaseMutex(udp_mutex);

			// Das Programm reagiert auf die Erkennung eines mfx-Dekoders und vergibt eine SID
			if ((passive_mode == 0) && ((can_msg.Id & 0xFFFF0000) == 0x00030000UL) && (can_msg.MsgLen == 5))
			{
				out_msg.MsgFlags = 0L;
				out_msg.MsgEFF = 1;
				out_msg.Id = 0x040301UL;
				out_msg.MsgLen = 6;
				out_msg.MsgData[0] = can_msg.MsgData[0];
				out_msg.MsgData[1] = can_msg.MsgData[1];
				out_msg.MsgData[2] = can_msg.MsgData[2];
				out_msg.MsgData[3] = can_msg.MsgData[3];
				out_msg.MsgData[4] = (nextmfxnr() >> 8) & 0x00FF;
				out_msg.MsgData[5] = lastmfxnr() & 0x00FF;
				WaitForSingleObject(can_mutex, INFINITE);
				CanTransmit(0, &out_msg, 1); // Fehler ignorieren
				ReleaseMutex(can_mutex);
				uid = can_msg.MsgData[3] | (can_msg.MsgData[2] << 8) | (can_msg.MsgData[1] << 16) | (can_msg.MsgData[0] << 24);
				udp_frame[0] = (out_msg.Id >> 24) & 0x000000FF;
				udp_frame[1] = (out_msg.Id >> 16) & 0x000000FF;
				udp_frame[2] = (out_msg.Id >> 8) & 0x000000FF;
				udp_frame[3] = out_msg.Id & 0x000000FF;
				udp_frame[4] = out_msg.MsgLen;
				for (i = 0; i < 8; ++i)
				{
					if (i < out_msg.MsgLen)
						udp_frame[5+i] = out_msg.MsgData[i];
					else
						udp_frame[5+i] = 0;
				}
				WaitForSingleObject(udp_mutex, INFINITE);
				sendto(sock_out, udp_frame, 13, 0, (struct sockaddr*)&serv_out, sizeof(serv_out));
				ReleaseMutex(udp_mutex);
				printf("\nNeuer mfx-Dekoder (UID %lX) angemeldet, Nummer: %d\n> ", uid, lastmfxnr());
			}
		}
	}
}

// Thread fuer UDP-Frame-Empfang
// Der Thread wartet in einer Schleife auf den Empfang von UDP-Frames und sendet die Botschaften
// dann ueber CAN zur Gleisbox.

void RxUDPFrameProc(void *pID)
{
	SOCKET sock_in;
	SOCKADDR_IN serv_in;
	char udp_frame[20];
	struct TCanMsg can_msg;
	int n, i;

	// Netzwerk-Verbindung fuer eingehende UDP-Frames oeffnen
	sock_in = socket(AF_INET, SOCK_DGRAM, 0);
	memset (&serv_in, 0, sizeof(serv_in));
	serv_in.sin_family = AF_INET;
	serv_in.sin_addr.s_addr = inet_addr(ipnumber_in());
	serv_in.sin_port = htons(15731);
	bind(sock_in, (SOCKADDR*)&serv_in, sizeof(serv_in));

	// Schleife, wartet auf eingehende UDP-Frames und gibt diese dann ueber CAN aus
	while(1)
    {
		n = recvfrom(sock_in, udp_frame, sizeof(udp_frame), 0, NULL, NULL);
		if (n == 13) // alle anderen UDP-Frames ignorieren
		{
			can_msg.MsgFlags = 0L;
			can_msg.MsgEFF = 1;
			can_msg.Id = (unsigned long)(((udp_frame[0] << 24) & 0xFF000000) |
				                         ((udp_frame[1] << 16) & 0x00FF0000) |
										 ((udp_frame[2] << 8)  & 0x0000FF00) |
										  (udp_frame[3]        & 0x000000FF));
			can_msg.MsgLen = udp_frame[4];
			for (i = 0; i < 8; ++i)
				can_msg.MsgData[i] = udp_frame[5+i];
			WaitForSingleObject(can_mutex, INFINITE);
			CanTransmit(0, &can_msg, 1); // Fehler ignorieren
			ReleaseMutex(can_mutex);
		}
	}
}

// Hauptprogramm

int _tmain(int argc, _TCHAR* argv[])
{
	int cur_argp;
	int cur_argc;
	WSADATA wsa; // Dummy fuer Winsock-Version, nicht benutzt
	unsigned int i;
	int err;
	char tmpstr[20];
	unsigned int sid;
	struct TCanMsg can_msg;
	char udp_frame[20];

	// Ausgabe der Programmversion
	printf("GBox2Eth 1.3\n");
	printf("Aufruf: gbox2eth [-p] [<COM-Kanal> <IP-Adresse> [<IP-Adresse (schreibend)>] ]\n");
	printf("Optionen:\n");
	printf(" -p  Passiv-Modus, mfx-Loks werden von GBox2Eth nicht automatisch angemeldet\n");

	// Einlesen der Konfigurationsdatei oder Auswerten der Kommandozeilenparameter
	cur_argc = argc;
	cur_argp = 1;
	passive_mode = 0;
	if ((cur_argc > 1) && (strlen ((char*)(argv[cur_argp])) == 2) && (argv[cur_argp][0] == '-') && (argv[cur_argp][1] == 'p'))
	{
		passive_mode = 1;
		cur_argc--;
		cur_argp++;
	}
	if (cur_argc == 3)
	{
		createcfgfile((char*)(argv[cur_argp]), (char*)(argv[cur_argp+1]));
		cmddescr();
	}
	if (cur_argc == 4)
	{
		createcfgfile((char*)(argv[cur_argp]), (char*)(argv[cur_argp+1]), (char*)(argv[cur_argp+2]));
		cmddescr();
	}
	else
	{
		if (opencfgfile())
		{
			cmddescr();
		}
		else
		{
			exit(-1);
		}
	}

	// Initialisieren der beiden Mutexe zum Schutz der Schreiboperationen auf UDP und CAN
	udp_mutex = CreateMutex(NULL, FALSE, NULL);
	can_mutex = CreateMutex(NULL, FALSE, NULL);

	// Initialisieren der Windows-Socket-Bibliothek fuer den Zugriff auf Ethernet/UDP
	WSAStartup(MAKEWORD(2,0), &wsa);

	// Netzwerkverbindung oeffnen fuer die ausgehenden UDP-Frames
	sock_out = socket(AF_INET, SOCK_DGRAM, 0);
	memset (&serv_out, 0, sizeof(serv_out));
	serv_out.sin_family = AF_INET;
	serv_out.sin_addr.s_addr = inet_addr(ipnumber_out());
	serv_out.sin_port = htons(15730);

	// Laden der CAN-Treiber-DLL und Oeffnen des CAN-Kanals
	if ((err = LoadDriver("mhstcan.dll")) < 0)
	{
		printf("LoadDriver failed (error code = %d)\n", err);
		CloseHandle(udp_mutex);
		CloseHandle(can_mutex);
		return 1;
	}
	if ((err = CanInitDriver(NULL)) < 0)
	{
		printf("CanInitDrv failed (error code = %d)\n", err);
		CloseHandle(udp_mutex);
		CloseHandle(can_mutex);
		return 1;
	}
	CanSetOptions("AutoConnect=1");
	CanSetRxEventCallback(&RxEvent); // Callback-Routine fuer CAN-Empfang
	CanSetEvents(EVENT_ENABLE_RX_MESSAGES);
	_snprintf_s (tmpstr, 20-1, "ComPort=%d", comchannel());
	if ((err = CanDeviceOpen(0, tmpstr)) < 0)
	{
		printf("CanDeviceOpen failed (error code = %d)\n", err);
		CloseHandle(udp_mutex);
		CloseHandle(can_mutex);
		return 1;
	}
	CanSetSpeed(0, CAN_250K_BIT);
	CanSetMode(0, OP_CAN_START, CAN_CMD_ALL_CLEAR);

	// Start eines eigenen Threads fuer UDP-Empfang
	_beginthread(RxUDPFrameProc, 0, NULL);

	// Sende Start-Kommando ueber CAN an die Gleisbox (dabei wird die Gleisbox auf "Go" gesetzt!)
	if (passive_mode == 0)
	{
		can_msg.MsgFlags = 0L;
		can_msg.MsgEFF = 1;
		can_msg.Id = 0x360301UL;
		can_msg.MsgLen = 5;
		can_msg.MsgData[0] = 0;
		can_msg.MsgData[1] = 0;
		can_msg.MsgData[2] = 0;
		can_msg.MsgData[3] = 0;
		can_msg.MsgData[4] = 0x11;
		WaitForSingleObject(can_mutex, INFINITE);
		CanTransmit(0, &can_msg, 1); // Fehler ignorieren
		ReleaseMutex(can_mutex);
	}

	// Bearbeitung der Kommandozeileneingaben (Endlosschleife)
	while (1)
	{
		switch (cmdinput ())
		{
		case 0: // Befehl: Ende
			CloseHandle(udp_mutex);
			CloseHandle(can_mutex);
			return 0;
		case 1: // Befehl: Letzte verwendete mfx-Nummer ausgeben
			if (lastmfxnr() > 0)
				printf("Letzte vergebene mfx-Nummer: %d\n", lastmfxnr());
			else
				printf("Noch keine mfx-Nummer vergeben.\n");
			break;
		case 2: // Befehl: Umschaltbefehl an mfx-Lok vorwaerts
			if (cmdparam() <= lastmfxnr())
			{
				can_msg.MsgFlags = 0L;
				can_msg.MsgEFF = 1;
				can_msg.Id = 0x0A0301UL;
				can_msg.MsgLen = 5;
				can_msg.MsgData[0] = 0;
				can_msg.MsgData[1] = 0;
				sid = cmdparam() + 16384;
				can_msg.MsgData[2] = (sid >> 8) & 0x00FF;
				can_msg.MsgData[3] = sid & 0x00FF;
				can_msg.MsgData[4] = 1;
				WaitForSingleObject(can_mutex, INFINITE);
				CanTransmit(0, &can_msg, 1); // Fehler ignorieren
				ReleaseMutex(can_mutex);
				udp_frame[0] = (can_msg.Id >> 24) & 0x000000FF;
				udp_frame[1] = (can_msg.Id >> 16) & 0x000000FF;
				udp_frame[2] = (can_msg.Id >> 8) & 0x000000FF;
				udp_frame[3] = can_msg.Id & 0x000000FF;
				udp_frame[4] = can_msg.MsgLen;
				for (i = 0; i < 8; ++i)
				{
					if (i < can_msg.MsgLen)
						udp_frame[5+i] = can_msg.MsgData[i];
					else
						udp_frame[5+i] = 0;
				}
				WaitForSingleObject(udp_mutex, INFINITE);
				sendto(sock_out, udp_frame, 13, 0, (struct sockaddr*)&serv_out, sizeof(serv_out));
				ReleaseMutex(udp_mutex);
				printf("Richtungswechsel (vorwaerts) fuer Lok %d durchgefuehrt\n", cmdparam());
			}
			else
			{
				printf("Richtungswechsel nicht moeglich, angegebene mfx-Nummer existiert nicht!\n");
			}
			break;
		case 3: // Befehl: Umschaltbefehl an mfx-Lok rueckwaerts
			if (cmdparam() <= lastmfxnr())
			{
				can_msg.MsgFlags = 0L;
				can_msg.MsgEFF = 1;
				can_msg.Id = 0x0A0301UL;
				can_msg.MsgLen = 5;
				can_msg.MsgData[0] = 0;
				can_msg.MsgData[1] = 0;
				sid = cmdparam() + 16384;
				can_msg.MsgData[2] = (sid >> 8) & 0x00FF;
				can_msg.MsgData[3] = sid & 0x00FF;
				can_msg.MsgData[4] = 2;
				WaitForSingleObject(can_mutex, INFINITE);
				CanTransmit(0, &can_msg, 1); // Fehler ignorieren
				ReleaseMutex(can_mutex);
				udp_frame[0] = (can_msg.Id >> 24) & 0x000000FF;
				udp_frame[1] = (can_msg.Id >> 16) & 0x000000FF;
				udp_frame[2] = (can_msg.Id >> 8) & 0x000000FF;
				udp_frame[3] = can_msg.Id & 0x000000FF;
				udp_frame[4] = can_msg.MsgLen;
				for (i = 0; i < 8; ++i)
				{
					if (i < can_msg.MsgLen)
						udp_frame[5+i] = can_msg.MsgData[i];
					else
						udp_frame[5+i] = 0;
				}
				WaitForSingleObject(udp_mutex, INFINITE);
				sendto(sock_out, udp_frame, 13, 0, (struct sockaddr*)&serv_out, sizeof(serv_out));
				ReleaseMutex(udp_mutex);
				printf("Richtungswechsel (rueckwaerts) fuer Lok %d durchgefuehrt\n", cmdparam());
			}
			else
			{
				printf("Richtungswechsel nicht moeglich, angegebene mfx-Nummer existiert nicht!\n");
			}
			break;
		default: // uups, kann eigentlich nicht sein -> ignorieren
			break;
		}
	}
}
