// serial - Kommunikation mit CC-Schnitte ueber virtuellen COM-Port
//
// (c) Stefan Krauss, 70437 Stuttgart
// 
// Lizenz: Nutzung und Weitergabe nach der GNU General Public License (GPL), Version 3
//         Es wird keinerlei Haftung oder Gewaehrleistung uebernommen, siehe GPL!

#include "stdafx.h"
#include "serial.h"

HANDLE hser;
OVERLAPPED olwrite;
OVERLAPPED olread;
void(*callback)(struct TCanMsg*);

// Thread fuer Empfang von Bytes der seriellen Schnittstelle
// Der Thread wartet in einer Schleife auf den Empfang von Bytes und ruft nach Erhalt von
// 13 Bytes (= eine komplette CAN-Botschaft) die Callback-Routine auf. Diese verarbeitet dann
// die empfangene CAN-Botschaft.

void RxSerProc(void *pID)
{
	unsigned char inbuffer[13];
	unsigned int inbytes;
	unsigned int i;
	struct TCanMsg msg;
	DWORD bytesread;
	unsigned char buf[2];
	DWORD err;

	// Overlap-Struktur fuer Empfang anlegen
	memset(&olread, 0, sizeof (OVERLAPPED));
	olread.hEvent = CreateEvent(NULL, TRUE, FALSE, NULL);

	// Empfang in Endlosschleife, Callback aufrufen, wenn kompletter CAN-Frame empfangen
	inbytes = 0;
	do
	{
		if (ReadFile(hser, buf, 1, &bytesread, &olread))
		{
			// Byte gelesen (ohne Warten, nicht overlapped)
			if (bytesread == 1)
				inbuffer[inbytes++] = buf[0];
			// else printf("0 Bytes gelesen (direkt)\n");
		}
		else
		{
			if (ERROR_IO_PENDING == (err = GetLastError()))
			{
				// Leseoperation overlapped
				if (WAIT_OBJECT_0 == (err = WaitForSingleObject(olread.hEvent, INFINITE)))
				{
					if (GetOverlappedResult(hser, &olread, &bytesread, FALSE))
					{
						// Byte gelesen
						if (bytesread == 1)
							inbuffer[inbytes++] = buf[0];
						// else printf("0 Bytes gelesen (overlapped)\n");
					}
					// else printf("Fehler bei GetOverlappedResult - Read (%d)\n", GetLastError());
				}
				// else printf("Fehler bei Wait (%d)\n", err);
			}
			// else printf("Fehler beim Lesen (%d)\n", err);
		}

		if (inbytes == 13)
		{
			msg.Id = (unsigned long)(((inbuffer[0] << 24) & 0xFF000000) |
				((inbuffer[1] << 16) & 0x00FF0000) |
				((inbuffer[2] << 8)  & 0x0000FF00) |
				(inbuffer[3]        & 0x000000FF));
			msg.MsgLen = inbuffer[4];
			for (i = 0; i < 8; ++i)
				msg.MsgData[i] = inbuffer[5+i];
			callback (&msg);
			inbytes = 0;
		}
	}
	while (1);
}

int ser_init(unsigned int comno, void(*cb)(struct TCanMsg*))
{
	char comstr[11];
	DCB dcb;
	COMMTIMEOUTS ct;

	callback = cb;

	// COM port oeffnen
	sprintf(comstr, "\\\\.\\COM%d", comno);
	hser = CreateFile(comstr, GENERIC_WRITE | GENERIC_READ, 0, NULL, OPEN_EXISTING, FILE_FLAG_OVERLAPPED, NULL);
	if (hser == INVALID_HANDLE_VALUE)
		printf("Fehler beim Oeffnen des COM-Ports (%d)\n", GetLastError());

	// Setzen der Uebertragungseinstellungen
	dcb.DCBlength = sizeof(DCB);
	GetCommState(hser, &dcb);
	dcb.BaudRate = 500000;
	dcb.ByteSize = 8;
	dcb.Parity = NOPARITY;
	dcb.StopBits = ONESTOPBIT;
	SetCommState(hser, &dcb);

	// Setzen der Timeouts (auf: keine Timeouts)
	GetCommTimeouts(hser, &ct);
	ct.ReadIntervalTimeout = 0;
	ct.ReadTotalTimeoutMultiplier = 0;
	ct.ReadTotalTimeoutConstant = 0;
	ct.WriteTotalTimeoutMultiplier = 0;
	ct.WriteTotalTimeoutConstant = 0;
	SetCommTimeouts(hser, &ct);

	// Puffer-Einstellungen
	SetupComm(hser, 130, 13);

	// Start eines eigenen Threads fuer Empfang
	_beginthread(RxSerProc, 0, NULL);

	// Overlap-Struktur fuer Sendeoperationen anlegen
	memset(&olwrite, 0, sizeof (OVERLAPPED));
	olwrite.hEvent = CreateEvent(NULL, TRUE, FALSE, NULL);

	return 0;
}

void ser_close()
{
	// sehr krudes Ende - Handles werden einfach geloescht
	// Achtung: das erzeugt ein paar Fehler in den Threads beim Ende des Programms,
	// die man aber ohne Fehlerausgaben nicht bemerkt
	CloseHandle(hser);
	CloseHandle(olwrite.hEvent);
	CloseHandle(olread.hEvent);
}

void ser_transmit(TCanMsg* msg)
{
	unsigned char buffer[13];
	unsigned int i;
	DWORD written;
	DWORD err;

	buffer[0] = (msg->Id >> 24) & 0x000000FF;
	buffer[1] = (msg->Id >> 16) & 0x000000FF;
	buffer[2] = (msg->Id >> 8) & 0x000000FF;
	buffer[3] = msg->Id & 0x000000FF;
	buffer[4] = msg->MsgLen;
	for (i = 0; i < 8; ++i)
	{
		if (i < msg->MsgLen)
			buffer[5+i] = msg->MsgData[i];
		else
			buffer[5+i] = 0;
	}
	if (WriteFile(hser, &buffer, 13, &written, &olwrite))
	{
		// WriteFile completed
	}
	else
	{
		if (ERROR_IO_PENDING == (err = GetLastError()))
		{
			if (WAIT_OBJECT_0 == (err = WaitForSingleObject(olwrite.hEvent, INFINITE)))
			{
				if (GetOverlappedResult(hser, &olwrite, &written, FALSE))
				{
					// WriteFile completed
				}
				else
					printf("Fehler bei GetOverlappedResult - Write (%d)\n", GetLastError());
			}
			else
				printf("Fehler beim Warten auf Ende Schreibvorgang COM-Port (%d)\n", err);
		}
		else
			printf("Fehler beim Schreiben auf den COM-Port (%d)\n", err);
	}
}

